/*
 *  InterruptableInputStream.java
 *  Copyright (C) 2005 Amin Ahmad. 
 *          
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 *  Amin Ahmad can be contacted at amin.ahmad@gmail.com or on the web at 
 *  www.ahmadsoft.org.
 */
package org.ahmadsoft.io.util;

import java.io.IOException;
import java.io.InputStream;

/**
 * An interruptable input stream. When an instance of this class
 * is interrupted via <code>setInterrupted</code>, subsequent 
 * attempts to read from it will fail.
 *  
 * @author Amin Ahmad
 */
public class InterruptableInputStream extends InputStream {
	private InputStream wrapped;
	private volatile boolean interrupted;

	public InterruptableInputStream(InputStream wrapped) {
		super();
		this.wrapped = wrapped;
		this.interrupted = false;
	}
	
	public void setInterrupted(boolean interrupted) {
		this.interrupted = interrupted;
	}

	public boolean isInterrupted() {
		return this.interrupted;
	}
	
	private void checkInterrupted() throws IOException {
		if (this.interrupted) {
			throw new IOException("Input is interrupted. Read failed.");
		}
	}
	public int available() throws IOException {
		checkInterrupted();
		return wrapped.available();
	}

	public void close() throws IOException {
		wrapped.close();
	}

	public boolean equals(Object obj) {
		return wrapped.equals(obj);
	}

	public int hashCode() {
		return wrapped.hashCode();
	}

	public void mark(int readlimit) {
		wrapped.mark(readlimit);
	}

	public boolean markSupported() {
		return wrapped.markSupported();
	}

	public int read() throws IOException {
		checkInterrupted();
		return wrapped.read();
	}

	public int read(byte[] b, int off, int len) throws IOException {
		checkInterrupted();
		return wrapped.read(b, off, len);
	}

	public int read(byte[] b) throws IOException {
		checkInterrupted();
		return wrapped.read(b);
	}

	public void reset() throws IOException {
		checkInterrupted();
		wrapped.reset();
	}

	public long skip(long n) throws IOException {
		checkInterrupted();
		return wrapped.skip(n);
	}

	public String toString() {
		return wrapped.toString();
	}
}
