/*
 *  RenderProgress.java
 *  Copyright (C) 2005 Amin Ahmad. 
 *          
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 *  Amin Ahmad can be contacted at amin.ahmad@gmail.com or on the web at 
 *  www.ahmadsoft.org.
 */
package org.ahmadsoft.foprocessor.ui.dialogs;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import org.ahmadsoft.foprocessor.FoProcessorPlugin;
import org.ahmadsoft.foprocessor.core.FileRenderSpecification;
import org.ahmadsoft.foprocessor.operations.ConversionOperation;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.internal.ui.preferences.IDebugPreferenceConstants;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.debug.ui.console.ConsoleColorProvider;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Shell;

/**
 * @author Amin Ahmad
 */
public class RenderProgress extends TitleAreaDialog implements IRunnableContext, ConversionOperation.Listener {

    private Image image = null;
    private ITextViewer console;
    private Document document;
    
    private ProgressBar progressBar;
    private Label consoleTitle;
    
    private ConsoleColorProvider colorProvider = new ConsoleColorProvider();
    
    private CountDownLatch createSignal = new CountDownLatch(1);
    
    private final int RUN_BACKGROUND_ID = 3141;
	private List<FileRenderSpecification> renderSpecs;
	
	private volatile boolean cancelled;
    
    /**
     * @param parentShell
     */
    public RenderProgress(Shell parentShell, List<FileRenderSpecification> renderSpecs) {
        super(parentShell);        
        setShellStyle(SWT.TITLE | SWT.BORDER | SWT.APPLICATION_MODAL | SWT.RESIZE);
        this.renderSpecs = renderSpecs;
    }
    
    protected Point getInitialSize() {
        return getShell().computeSize(460, 500, true);
    }
    
    protected Control createContents(Composite parent) {
        Control result = super.createContents(parent);
        
        createSignal.countDown();
        
        return result;
    }

    protected void createButtonsForButtonBar(Composite parent) {
//        Button btnBackground = createButton(parent, RUN_BACKGROUND_ID, "Run in &Background",
//                false);
//        btnBackground.setEnabled(false);
    	super.createButtonsForButtonBar(parent);
    }
    
    protected Control createDialogArea(Composite parent) {
        Composite compositeParent = (Composite)super.createDialogArea(parent);
        
        // Setup the title area
        //
        setTitleImage(getImage());
        setTitle("Rendering"); 
        setMessage("Rendering documents."); 
        
        Composite pageContainer = new Composite(compositeParent, SWT.NONE);
        GridData gd = new GridData(GridData.FILL_BOTH);
        pageContainer.setLayoutData(gd);
        pageContainer.setFont(parent.getFont());       
        
        // Document 
        //
        document = new Document();
        
        //layout for page
        //
        GridLayout layout = new GridLayout();
        layout.numColumns = 1;
        
        pageContainer.setLayout(layout);
        
        progressCaption = new Label(pageContainer, SWT.NONE);
        progressCaption.setText("Progress:");
        progressCaption.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        progressBar = new ProgressBar(pageContainer, SWT.SMOOTH | SWT.HORIZONTAL);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.heightHint = 10;
        progressBar.setLayoutData(gd);
        
        consoleTitle = new Label(pageContainer, SWT.NONE);
        consoleTitle.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        console = new TextViewer(pageContainer, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
        console.getTextWidget().setFont(JFaceResources.getFont(IDebugPreferenceConstants.CONSOLE_FONT));
        console.getTextWidget().setLayoutData(new GridData(GridData.FILL_BOTH));
        console.setDocument(document);
         
        // stream.setColor(fColorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));

//        btnAlwaysBackground = new Button(pageContainer, SWT.CHECK);
//		btnAlwaysBackground.setText("Always render in the background.");
		
        btnBuildAutomatically = new Button(pageContainer, SWT.CHECK);
		btnBuildAutomatically.setText("Integrate rendering for this document into the build cycle.");

        boolean buildSelVal = true;
        for (FileRenderSpecification spec: renderSpecs) {
            try {
				if (!FoProcessorPlugin.getDefault().isAutoBuild(spec.getInputFile())) {
					buildSelVal = false;
					break;
				}
			} catch (CoreException e) {
				buildSelVal = false;
				break;
			}
        }
        
        btnBuildAutomatically.setSelection(buildSelVal);
        
        // Build the separator line
        //
        Label separator= new Label(compositeParent, SWT.HORIZONTAL | SWT.SEPARATOR);
        separator.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        getShell().setText("Apache FOP 0.92beta");
        
        return compositeParent;
    }

    public Image getImage() {
        if (image == null) {
            image = FoProcessorPlugin.getDefault().getImageDescriptor("full/wizban/RenderProgress.gif").createImage();
        }
        return image;
    }
    
    public void setConsoleTitle() {
        SimpleDateFormat df = new SimpleDateFormat("MMM dd, yyyy HH:mm:ss");
        consoleTitle.setText("XSL-FO Rendering [Powered by Apache FOP] (" +df.format(new Date())+ ")");
    }
    
    /**
     * Renders the document. Should not be run on the UI thread.
     * 
     * @param mimeType
     * @param extension
     * @param resources
     */
    public void beginRendering() {
    	    	
        try {
            createSignal.await();
        } catch (InterruptedException e) {
            error(null, e);
        }
        
        // Setup the FOP Logger
        //

        Handler lh = new LogHandler();
        Logger.getLogger("org.apache.fop").addHandler(lh);
        
        
        // Set the titles and the button states
        //
        setConsoleTitle();
        
        Button okButton = getButton(IDialogConstants.OK_ID);
        if (okButton != null) {
            okButton.setEnabled(false);
        }
        
        Button cnButton = getButton(IDialogConstants.CANCEL_ID);
        if (cnButton != null) {
            cnButton.setEnabled(true);
        }
        
        Button bkgButton = getButton(RUN_BACKGROUND_ID);
        if (bkgButton != null) {
        	bkgButton.setEnabled(true);
        }

        conversionOp = new ConversionOperation(renderSpecs, null, RenderProgress.this);
		IRunnableWithProgress runnable = new IRunnableWithProgress() {
            public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                conversionOp.setProgressMonitor(monitor);
                if (!cancelled) {
                	conversionOp.run();
                }
            } 
        };
        
        try {
            run(true, true, runnable);
        } catch (Exception e) {
            error(null, e);
        }
    }

    
    /**
     * @inheritDoc
     */
    public boolean close() {
        getImage().dispose();
        return super.close();
    }
    /**
     * @inheritDoc
     */
    public void cancelPressed() {
    	cancelled = true;
    	if (conversionOp != null) {
    		conversionOp.setInterrupted();
    	}
    	super.cancelPressed();
    }
    
    /**
     * @inheritDoc
     */
    public void okPressed() {
    	// Process builder.
    	//
    	boolean buildValue = btnBuildAutomatically.getSelection();
    	
    	for (FileRenderSpecification spec: renderSpecs) {
            IProject project = spec.getInputFile().getProject();
    
            try {
                if (buildValue == true) {
                	FoProcessorPlugin.getDefault().addFopNature(project);
                } 
            	FoProcessorPlugin.getDefault().setAutoBuild(spec.getInputFile(), buildValue, spec);
            } catch (CoreException e) {
            	e.printStackTrace();
            }
        }
    	
    	super.okPressed();
    }

    // IRunnableContext
    //
    
    public void run(boolean fork, boolean cancelable, IRunnableWithProgress runnable) throws InvocationTargetException, InterruptedException {
        // Create the Progress Monitor
        //
        ProgressMonitor monitor = new ProgressMonitor();
        
        ModalContext.run(runnable, fork, monitor, getShell().getDisplay());
    }
    
    // Logging Methods --------------------------------------------------------

    private static final String nl = System.getProperty("line.separator");
    private Label progressCaption;
	private Button btnAlwaysBackground;
	private Button btnBuildAutomatically;
	private ConversionOperation conversionOp;
    
    private String toString(Throwable t) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        PrintStream ps = new PrintStream(bos);
        t.printStackTrace(ps);
        ps.close();
        
        return new String(bos.toByteArray());
    }
    
    private void append(final String s, final Color color) {
        if (s == null) {
            return;
        }
        Display.getDefault().asyncExec(
            new Runnable() {
                public void run() {
                    try {
                        StyleRange styleRange = new StyleRange();
                        styleRange.start = document.getLength() - 1;
                        styleRange.length = s.length() + nl.length();
                        styleRange.foreground = color;
                        
                        document.replace(document.getLength(), 0, s);
                        document.replace(document.getLength(), 0, nl);

                        if (console.getTextWidget() != null) { // amazingly, this happens...
                            console.getTextWidget().setStyleRange(styleRange);
                        }
                    } catch (BadLocationException e) {
                        e.printStackTrace();
                    }
                }
            }
        );
    }

    /**
     * @inheritdoc
     */
    public void error(String arg0) {
        append(arg0, colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
    }

    /**
     * @inheritdoc
     */
    public void error(String arg0, Throwable arg1) {
        append(arg0, colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
        append(toString(arg1), colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
        
    }

    /**
     * @inheritdoc
     */
    public void fatalError(String arg0) {
        append(arg0, colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
    }

    /**
     * @inheritdoc
     */
    public void fatalError(String arg0, Throwable arg1) {
        append(arg0, colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
        append(toString(arg1), colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM));
    }
    
    public class LogHandler extends Handler {

        @Override
        public void publish(LogRecord record) {
            Color color = null;
            if (record.getLevel().intValue() <= Level.INFO.intValue()) {
                color = colorProvider.getColor(IDebugUIConstants.ID_STANDARD_OUTPUT_STREAM);
            } else {
                color = colorProvider.getColor(IDebugUIConstants.ID_STANDARD_ERROR_STREAM);
            } 
            append(record.getMessage(), color);
            if (record.getThrown() != null) {
                append(RenderProgress.this.toString(record.getThrown()), color);
            }
        }

        @Override
        public void flush() {
            // TODO Auto-generated method stub
            
        }

        @Override
        public void close() throws SecurityException {
            // TODO Auto-generated method stub
            
        }
        
    }
    
    // Progress Monitor -------------------------------------------------------
    
    public class ProgressMonitor implements IProgressMonitor {

        private String name;
        private boolean canceled;

        /**
         * @inheritdoc
         */
        public void beginTask(final String name, final int totalWork) {
            Display.getDefault().asyncExec(
                new Runnable() {
                    public void run() {
                        progressBar.setMinimum(0);
                        progressBar.setSelection(0);
                        progressBar.setMaximum(totalWork);

                        progressCaption.setText("Progress: " + name);
                        ProgressMonitor.this.name = name;
                    }
                }
            );
        }

        /**
         * @inheritdoc
         */
        public void done() {
            Display.getDefault().asyncExec(
                new Runnable() {
                    public void run() {
                        progressBar.setSelection(progressBar.getMaximum());
                        progressCaption.setText("Progress: Done");
                    }
                }
            );
        }

        /**
         * @inheritdoc
         */
        public void internalWorked(double work) {
            worked((int) Math.round(work));
        }

        /**
         * @inheritdoc
         */
        public boolean isCanceled() {
            return canceled;
        }

        /**
         * @inheritdoc
         */
        public void setCanceled(boolean value) {
            this.canceled = value;
            if (value) {
                Display.getDefault().asyncExec(
                    new Runnable() {
                        public void run() {
                            progressCaption.setText("Progress: " + ProgressMonitor.this.name + "... Requesting cancellation");
                        }
                    }
                );
            }
        }

        /**
         * @inheritdoc
         */
        public void setTaskName(String name) {
            this.name = name;
        }

        /**
         * @inheritdoc
         */
        public void subTask(final String name) {
            Display.getDefault().asyncExec(
                new Runnable() {
                    public void run() {
                        progressCaption.setText("Progress: " + ProgressMonitor.this.name + "... " + name);
                    }
                }
            );
        }

        /**
         * @inheritdoc
         */
        public void worked(final int work) {
            Display.getDefault().asyncExec(
                new Runnable() {
                    public void run() {
                        progressBar.setSelection(progressBar.getSelection() + work);
                    }
                }
            );
        }
        
    }

	public void onFinish() {
        Display.getDefault().asyncExec(
            new Runnable() {
                public void run() {
                    Button okButton = getButton(IDialogConstants.OK_ID);
                    if (okButton != null) {
                        okButton.setEnabled(true);
                        okButton.setFocus();
                    }
                    
                    Button cnButton = getButton(IDialogConstants.CANCEL_ID);
                    if (cnButton != null) {
                        cnButton.setEnabled(false);
                    }
                    
                    Button bkgButton = getButton(RUN_BACKGROUND_ID);
                    if (bkgButton != null) {
                    	bkgButton.setEnabled(false);
                    }
                }
            }
        );
	}

	public void onError(String msg, Throwable throwable) {
		error(msg, throwable);
	}
}
