/*
 *  FileRenderSpecification.java
 *  Copyright (C) 2005 Amin Ahmad. 
 *          
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 *  Amin Ahmad can be contacted at amin.ahmad@gmail.com or on the web at 
 *  www.ahmadsoft.org.
 */
package org.ahmadsoft.foprocessor.core;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

/**
 * Encapsulates a specification for rendering a file.
 */
public class FileRenderSpecification implements Externalizable {
	private IFile inputFile;
	private IPath outputPath;
	private String mimeType;
	private Map<String, Serializable> properties 
		= new HashMap<String, Serializable>();
	
	public IFile getInputFile() {
		return inputFile;
	}
	public void setInputFile(IFile inputFile) {
		this.inputFile = inputFile;
	}
	public String getMimeType() {
		return mimeType;
	}
	public void setMimeType(String mimeType) {
		this.mimeType = mimeType;
	}
	public IPath getOutputPath() {
		return outputPath;
	}
	public void setOutputFile(IPath outputPath) {
		this.outputPath = outputPath;
	}
	
	public static IPath computeOutputFile(IFile inputFile, String extension) {
		IPath fullPath = inputFile.getLocation();
		IPath path = fullPath.removeFileExtension().addFileExtension(extension);
		return path;
	}
	public void writeExternal(ObjectOutput out) throws IOException {
		
		// do not externalize the input file. Rather, the render spec is 
		// expected to be stored with the input file.
		
		out.writeUTF(outputPath.toPortableString());
		out.writeUTF(mimeType);
		out.writeObject(properties);
	}
	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		outputPath = Path.fromPortableString(in.readUTF());
		mimeType = in.readUTF();
		properties = (Map<String, Serializable>) in.readObject();
	}
	
	/**
	 * Serializes a file render spec into a string. Returns
	 * <code>null</code> on unexpected failure.
	 * 
	 * @param spec
	 * @return
	 */
	public static String toString(FileRenderSpecification spec) {
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		try {
			ObjectOutputStream oos = new ObjectOutputStream(bos);
			
			oos.writeObject(spec);
			oos.close();
			return Base64.byteArrayToBase64(bos.toByteArray());
		} catch (IOException e) {
			return null;
		}
	}
	public static FileRenderSpecification fromString(IFile inputFile, String spec) {
		if (spec == null) {
			return null;
		}
		
		byte[] specBytes = Base64.base64ToByteArray(spec);
		ByteArrayInputStream bis = new ByteArrayInputStream(specBytes);
		try {
			ObjectInputStream ois = new ObjectInputStream(bis);
			
			FileRenderSpecification result = (FileRenderSpecification) ois.readObject();
			result.setInputFile(inputFile);
			return result;
		} catch (Exception e) {	// IO, Class not found, etc..
			return null;
		}
	}
}
